<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ClienteController extends Controller
{
    /**
     * Mostrar la lista de clientes con un buscador.
     */
    public function index(Request $request)
    {
        $query = Cliente::query();

        if ($request->filled('buscar')) {
            $buscar = $request->input('buscar');
            $query->where('nombre', 'like', "%$buscar%")
                ->orWhere('apellidos', 'like', "%$buscar%")
                ->orWhere('carnet', 'like', "%$buscar%")
                ->orWhere('telefono', 'like', "%$buscar%");
        }

        $clientes = $query->paginate(10);

        return view('clientes.index', compact('clientes'));
    }
    public function create()
{
    return view('clientes.create');
}
    public function edit($id)
    {
        $cliente = Cliente::findOrFail($id); // Busca al cliente por su ID o lanza un error 404 si no existe
        return view('clientes.edit', compact('cliente'));
    }
    /**
     * Almacenar un nuevo cliente.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nombre' => 'required|regex:/^[a-zA-ZñÑáéíóúÁÉÍÓÚ\'\s]+$/u|max:100',
            'apellidos' => 'required|regex:/^[a-zA-ZñÑáéíóúÁÉÍÓÚ\'\s]+$/u|max:100',
   'carnet' => 'required|unique:clientes,carnet|min:6|max:15|regex:/^\d+$/',

            'telefono' => 'nullable|digits:8',
            'observacion' => 'nullable|max:255',
        ], [
            'nombre.regex' => 'El nombre solo puede contener letras, espacios, ñ y apóstrofes.',
            'apellidos.regex' => 'Los apellidos solo pueden contener letras, espacios, ñ y apóstrofes.',
            'telefono.digits' => 'El teléfono debe contener exactamente 8 dígitos.',
            'carnet.required' => 'El campo carnet es obligatorio.',
            'carnet.unique' => 'El carnet ya está registrado en el sistema.',
            'carnet.min' => 'El carnet debe tener al menos 6 dígitos.',
            'carnet.max' => 'El carnet no puede tener más de 15 dígitos.',
            'carnet.regex' => 'El carnet debe contener únicamente números.',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        Cliente::create([
            'nombre' => $request->nombre,
            'apellidos' => $request->apellidos,
            'carnet' => $request->carnet,
            'telefono' => $request->telefono,
            'observacion' => $request->observacion,
            'estado' => 1, // Por defecto el estado es activo (1)
        ]);

        return redirect()->route('clientes.index')->with('success', 'Cliente registrado exitosamente.');
    }

    /**
     * Actualizar un cliente existente.
     */
    public function update(Request $request, Cliente $cliente)
    {
        $validator = Validator::make($request->all(), [
            'nombre' => 'required|regex:/^[a-zA-ZñÑáéíóúÁÉÍÓÚ\'\s]+$/u|max:100',
            'apellidos' => 'required|regex:/^[a-zA-ZñÑáéíóúÁÉÍÓÚ\'\s]+$/u|max:100',
           'carnet' => 'required|unique:clientes,carnet,' . $cliente->id . '|min:6|max:15|regex:/^\d+$/',

            'telefono' => 'nullable|digits:8',
            'observacion' => 'nullable|max:255',
        ], [
            'nombre.regex' => 'El nombre solo puede contener letras, espacios, ñ y apóstrofes.',
            'apellidos.regex' => 'Los apellidos solo pueden contener letras, espacios, ñ y apóstrofes.',
            'telefono.digits' => 'El teléfono debe contener exactamente 8 dígitos.',
            'carnet.required' => 'El campo carnet es obligatorio.',
            'carnet.unique' => 'El carnet ya está registrado en el sistema.',
            'carnet.min' => 'El carnet debe tener al menos 6 dígitos.',
            'carnet.max' => 'El carnet no puede tener más de 15 dígitos.',
            'carnet.regex' => 'El carnet debe contener únicamente números.',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $cliente->update([
            'nombre' => $request->nombre,
            'apellidos' => $request->apellidos,
            'carnet' => $request->carnet,
            'telefono' => $request->telefono,
            'observacion' => $request->observacion,
        ]);

        return redirect()->route('clientes.index')->with('success', 'Cliente actualizado exitosamente.');
    }

    /**
     * Cambiar el estado de un cliente (activar/desactivar).
     */
    public function destroy(Cliente $cliente)
    {
        $cliente->estado = !$cliente->estado;
        $cliente->save();

        $mensaje = $cliente->estado ? 'Cliente activado exitosamente.' : 'Cliente desactivado exitosamente.';
        return redirect()->route('clientes.index')->with('success', $mensaje);
    }
}
